/*****************************************************************************
 * prodos/inode.c
 * Inode operations.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 22-May-2001: Created
 *****************************************************************************/

/*= Kernel Includes =========================================================*/

#include <linux/blkdev.h>

/*= ProDOS Includes =========================================================*/

#include "prodos.h"

/*= Forward Declarations ====================================================*/

/*= Interface Functions =====================================================*/

/*****************************************************************************
 * prodos_read_inode()
 *****************************************************************************/
void prodos_read_inode(struct inode *inode) {
	SHORTCUT struct prodos_inode_info * const pi = PRODOS_I(inode);
	SHORTCUT struct super_block * const sb = inode->i_sb;
	SHORTCUT const int dblk = PRODOS_INO_DBLK(inode->i_ino);
	struct buffer_head *bh = NULL;

	/* Initialize the inode structure. */
	memset(pi,0,sizeof(struct prodos_inode_info));

	/* Read the directory block containing this entry. */
	bh = prodos_bread(sb,dblk);
	if (!bh) {
		PRODOS_ERROR_0(sb,"prodos_bread() failed");
		make_bad_inode(inode);
		goto out_cleanup;
	}

	/* Read a directory inode. */
	if (PRODOS_INO_DENT(inode->i_ino) == PRODOS_DENT_DIR) {
		struct prodos_dir_block_first *dir = NULL;

		/* Get a pointer to the directory block. */
		dir = (struct prodos_dir_block_first*)bh->b_data;

		/* Initialize items that are common to all directories. */
		pi->i_key = PRODOS_INO_DBLK(inode->i_ino);
		inode->i_op = &prodos_dir_inode_operations;
		inode->i_fop = &prodos_dir_operations;
		inode->i_mode = S_IFDIR;

		/* Set items that are specific to the volume directory. */
		if (dblk == PRODOS_VOLUME_DIR_BLOCK) {
			/* Get the various timestamps. */
			inode->i_mtime = inode->i_atime = inode->i_ctime =
			 prodos_get_time(le16_to_cpu(dir->u.vol_header.create_date),
			  le16_to_cpu(dir->u.vol_header.create_time));

			/* Convert the access flags. */
			inode->i_mode |= prodos_get_mode(dir->u.vol_header.access,1);

			/* Volume directory should always be 4 blocks in length. */
			inode->i_blocks = PRODOS_VOLUME_DIR_LENGTH;
		}

		/* Set items that are specific to subdirectories. */
		else {
			struct buffer_head *parent_bh = NULL;
			struct prodos_dir_entry *parent_ent = NULL;

			/* Load parent directory block. */
			parent_bh = prodos_bread(sb,le16_to_cpu(dir->u.dir_header.parent_block));
			if (!parent_bh) {
				PRODOS_ERROR_0(sb,"failed to load parent directory block");
				make_bad_inode(inode);
				goto out_cleanup;
			}
			parent_ent = &((struct prodos_dir_block*)parent_bh->b_data)->entries[dir->u.dir_header.parent_entry - 1];

			/* Get the various timestamps. */
			inode->i_mtime = inode->i_mtime = inode->i_ctime =
			 prodos_get_time(le16_to_cpu(dir->u.dir_header.create_date),
			  le16_to_cpu(dir->u.dir_header.create_time));

			/* Convert the access flags. */
			inode->i_mode |= prodos_get_mode(dir->u.dir_header.access,1);

			/* Get the number of blocks used. */
			inode->i_blocks = le16_to_cpu(parent_ent->blocks_used);

			/* Release parent directory block. */
			prodos_brelse(parent_bh);
		}

		/* Calculate the "size" of the directory. */
		inode->i_size = inode->i_blocks << PRODOS_BLOCK_SIZE_BITS;
	}

	/* Read a file inode. */
	else {
		struct prodos_dir_entry *ent = NULL;

		/* Get a pointer to the entry. */
		ent = &((struct prodos_dir_block*)bh->b_data)->entries[PRODOS_INO_DENT(inode->i_ino)];

		/* Set up stuff that is specific to files. */
		pi->i_filetype = ent->file_type;
		pi->i_stype = PRODOS_GET_STYPE(ent->name);
		pi->i_key = le16_to_cpu(ent->key_block);
		pi->i_auxtype = le16_to_cpu(ent->aux_type);
		pi->i_access = ent->access;
		inode->i_op = &prodos_file_inode_operations;
		inode->i_fop = &prodos_file_operations;
		inode->i_mapping->a_ops = &prodos_address_space_operations;
		inode->i_mode = S_IFREG | prodos_get_mode(ent->access,0);
		inode->i_mtime = prodos_get_time(ent->mod_date,ent->mod_time);
		inode->i_atime = inode->i_ctime = prodos_get_time(
		 le16_to_cpu(ent->create_time),le16_to_cpu(ent->create_date));

		/* Will this file need CR->LF conversion? */
		if ((PRODOS_INO_DFORK(inode->i_ino) == PRODOS_DFORK_DATA) &&
		 (PRODOS_SB(sb)->s_flags & PRODOS_FLAG_CONVERT_CR) &&
		 strchr(PRODOS_TEXT_TYPES,ent->file_type))
			pi->i_flags |= PRODOS_I_CRCONV;

		/* Determine file size for non-extended files. */
		if (!PRODOS_ISEXTENDED(*ent)) {
			inode->i_size = le24_to_cpup(&ent->eof);
			inode->i_blocks = le16_to_cpu(ent->blocks_used);

			/* "Meta" files have an additional block of header data. */
			if (PRODOS_INO_DFORK(inode->i_ino) == PRODOS_DFORK_META) {
				inode->i_size += PRODOS_BLOCK_SIZE;
				inode->i_blocks++;
			}
		}

		/* Determine file size for extended files. */
		else {
			struct buffer_head *ebh = NULL;
			struct prodos_ext_key_block *ext_key = NULL;

			/* Read the file's extended directory block. */
			ebh = prodos_bread(sb,le16_to_cpu(ent->key_block));
			if (!ebh) {
				PRODOS_ERROR_0(sb,"failed to read extended key block");
				make_bad_inode(inode);
				goto out_cleanup;
			}
			ext_key = (struct prodos_ext_key_block*)ebh->b_data;

			/* Calculate size of the appropriate file or pseudo-file. */
			switch PRODOS_INO_DFORK(inode->i_ino) {
				case PRODOS_DFORK_DATA:
					inode->i_blocks = le16_to_cpu(ext_key->data.blocks_used);
					inode->i_size = le24_to_cpup(&ext_key->data.eof);
					break;
				case PRODOS_DFORK_META:
					inode->i_blocks =
					 le16_to_cpu(ext_key->data.blocks_used) +
					 le16_to_cpu(ext_key->res.blocks_used);
					inode->i_size = 512 +
					 ((le24_to_cpup(&ext_key->data.eof) + 511) & 0xfffffe00) +
					 ((le24_to_cpup(&ext_key->res.eof) + 511) & 0xfffffe00);
					break;
				case PRODOS_DFORK_RES:
					inode->i_blocks = le16_to_cpu(ext_key->res.blocks_used);
					inode->i_size = le24_to_cpup(&ext_key->res.eof);
					break;
			}

			/* Release the extended directory block. */
			prodos_brelse(ebh);
			ebh = NULL;
		}
	}

	/* Set up stuff that is common to every type of inode. */
	inode->i_nlink = 1;
	inode->i_uid = 0; /* root uid (just for now.) */
	inode->i_gid = 0; /* root gid (just for now.) */

out_cleanup:
	/* Release the directory block. */
	prodos_brelse(bh);
	bh = NULL;
}

/*****************************************************************************
 * prodos_write_inode()
 * Write an @inode's metainformation back to disk.
 *****************************************************************************/
void prodos_write_inode(struct inode *inode,int unused) {
	SHORTCUT struct prodos_inode_info * const pi = PRODOS_I(inode);
	struct buffer_head *dir_bh = NULL;
	struct prodos_dir_entry *ent = NULL;
	u32 eof = cpu_to_le32((u32)inode->i_size);

	PRODOS_INFO_1(inode->i_sb,"called for inode 0x%08x",(int)inode->i_ino);

	/* XXX: we'll eventually want to modify privileges and time stamps. */

	/* Do nothing if the inode is not linked (free.) */
	if (!inode->i_nlink) return;

	/* Load the directory block and get pointer to this entry. */
	dir_bh = prodos_bread(inode->i_sb,PRODOS_INO_DBLK(inode->i_ino));
	if (!dir_bh) {
		PRODOS_ERROR_0(inode->i_sb,"failed to read directory block");
		goto out_cleanup;
	}

	/* Write a directory inode. */
	if (PRODOS_INO_DENT(inode->i_ino) == PRODOS_DENT_DIR) {
		struct prodos_dir_block_first *dir = (void*)dir_bh->b_data;
		struct buffer_head *parent_bh = NULL;

		/* Load and update the parent directory entry. */
		parent_bh = prodos_bread(inode->i_sb,le16_to_cpu(dir->u.dir_header.parent_block));
		if (!parent_bh) {
			PRODOS_ERROR_0(inode->i_sb,"failed to read parent directory block");
			goto out_cleanup;
		}
		ent = &((struct prodos_dir_block*)parent_bh->b_data)->entries[dir->u.dir_header.parent_entry - 1];
		memcpy(&ent->eof,&eof,3);
		ent->blocks_used = cpu_to_le16(inode->i_blocks);
		mark_buffer_dirty(parent_bh);
		prodos_brelse(parent_bh);
	}

	/* Write a file inode. */
	else {
		/* Update the directory entry. */		
		ent = &((struct prodos_dir_block*)dir_bh->b_data)->entries[PRODOS_INO_DENT(inode->i_ino)];
		memcpy(&ent->eof,&eof,3);
		ent->key_block = cpu_to_le16(pi->i_key);
		ent->blocks_used = cpu_to_le16(inode->i_blocks);
		ent->name[0] = (ent->name[0] & 0x0f) | pi->i_stype;
		mark_buffer_dirty(dir_bh);
	}

out_cleanup:
	if (dir_bh) prodos_brelse(dir_bh);
}

/*****************************************************************************
 * prodos_put_inode()
 *****************************************************************************/
void prodos_put_inode(struct inode *inode) {
	PRODOS_INFO_1(inode->i_sb,"called on ino 0x%08x",(int)inode->i_ino);
}

